<?php
/**
 * PHP Email Requirements Checker
 * This script checks if your PHP environment has all requirements for sending emails
 */

// Function to check if extension is loaded
function checkExtension($extension) {
    return extension_loaded($extension);
}

// Function to check PHP version
function checkPHPVersion() {
    return version_compare(PHP_VERSION, '7.0', '>=');
}

// Function to check ini setting
function checkIniSetting($setting) {
    return ini_get($setting);
}

// Function to display status
function displayStatus($check, $description, $recommendation = '') {
    $status = $check ? '✅ PASS' : '❌ FAIL';
    echo sprintf("%-50s %s\n", $description, $status);
    if (!$check && $recommendation) {
        echo "   → Recommendation: $recommendation\n";
    }
}

echo "=== PHP EMAIL REQUIREMENTS CHECKER ===\n\n";

// PHP Version Check
echo "1. PHP VERSION:\n";
displayStatus(checkPHPVersion(), "PHP 7.0 or higher", "Update PHP to 7.0 or higher");
echo "   Current version: " . PHP_VERSION . "\n\n";

// Required Extensions
echo "2. REQUIRED PHP EXTENSIONS:\n";
displayStatus(checkExtension('openssl'), "OpenSSL Extension", "Enable openssl extension in php.ini");
displayStatus(checkExtension('curl'), "cURL Extension", "Enable curl extension in php.ini");
displayStatus(checkExtension('sockets'), "Sockets Extension", "Enable sockets extension in php.ini");
displayStatus(checkExtension('mbstring'), "Multibyte String Extension", "Enable mbstring extension in php.ini");
displayStatus(checkExtension('iconv'), "Iconv Extension", "Enable iconv extension in php.ini");

// Optional but recommended extensions
echo "\n3. OPTIONAL EXTENSIONS (Recommended):\n";
displayStatus(checkExtension('intl'), "Internationalization Extension", "Enable intl extension for better Unicode support");
displayStatus(checkExtension('filter'), "Filter Extension", "Enable filter extension for email validation");
displayStatus(checkExtension('hash'), "Hash Extension", "Enable hash extension for security");

// PHP Configuration
echo "\n4. PHP CONFIGURATION:\n";
$maxExecutionTime = ini_get('max_execution_time');
displayStatus($maxExecutionTime >= 30, "Max Execution Time (≥30s)", "Set max_execution_time = 30 or higher");
echo "   Current: " . ($maxExecutionTime == 0 ? 'unlimited' : $maxExecutionTime . 's') . "\n";

$memoryLimit = ini_get('memory_limit');
displayStatus(true, "Memory Limit", "");
echo "   Current: $memoryLimit\n";

$allowUrlFopen = ini_get('allow_url_fopen');
displayStatus($allowUrlFopen, "Allow URL fopen", "Set allow_url_fopen = On");

$fileUploads = ini_get('file_uploads');
displayStatus($fileUploads, "File Uploads (for attachments)", "Set file_uploads = On");

// Error Reporting
echo "\n5. ERROR REPORTING:\n";
$logErrors = ini_get('log_errors');
displayStatus($logErrors, "Log Errors", "Set log_errors = On");

$displayErrors = ini_get('display_errors');
displayStatus(!$displayErrors, "Display Errors (should be Off)", "Set display_errors = Off for production");

// Check if PHPMailer is available
echo "\n6. PHPMAILER AVAILABILITY:\n";
$composerAutoload = file_exists(__DIR__ . '/../vendor/autoload.php');
displayStatus($composerAutoload, "Composer Autoloader", "Run 'composer install' to install dependencies");

if ($composerAutoload) {
    require_once __DIR__ . '/../vendor/autoload.php';
    $phpmailerAvailable = class_exists('PHPMailer\\PHPMailer\\PHPMailer');
    displayStatus($phpmailerAvailable, "PHPMailer Class", "Install PHPMailer via Composer");
}

// Network connectivity check
echo "\n7. NETWORK CONNECTIVITY:\n";

// Check if we can resolve DNS
$dnsCheck = gethostbyname('gmail.com') !== 'gmail.com';
displayStatus($dnsCheck, "DNS Resolution", "Check your internet connection and DNS settings");

// Function to check port connectivity
function checkPort($host, $port, $timeout = 5) {
    $connection = @fsockopen($host, $port, $errno, $errstr, $timeout);
    if (is_resource($connection)) {
        fclose($connection);
        return true;
    }
    return false;
}

// Check common SMTP ports
$gmailSMTP = checkPort('smtp.gmail.com', 587, 5);
displayStatus($gmailSMTP, "Gmail SMTP Port 587", "Check firewall settings or try port 465");

$outlookSMTP = checkPort('smtp.office365.com', 587, 5);
displayStatus($outlookSMTP, "Outlook SMTP Port 587", "Check firewall settings");

// Check localhost SMTP (for local servers)
$localSMTP = checkPort('localhost', 25, 2);
displayStatus($localSMTP, "Localhost SMTP Port 25 (optional)", "Install local SMTP server if needed");

// Summary
echo "\n=== SUMMARY ===\n";
echo "If all checks pass, your PHP environment is ready for sending emails.\n";
echo "If any checks fail, follow the recommendations above.\n\n";

// Configuration examples
echo "=== CONFIGURATION EXAMPLES ===\n\n";

echo "1. XAMPP php.ini location (Windows):\n";
echo "   C:\\xampp\\php\\php.ini\n\n";

echo "2. Required php.ini settings:\n";
echo "   extension=openssl\n";
echo "   extension=curl\n";
echo "   extension=sockets\n";
echo "   extension=mbstring\n";
echo "   extension=iconv\n";
echo "   max_execution_time = 30\n";
echo "   memory_limit = 128M\n";
echo "   allow_url_fopen = On\n";
echo "   file_uploads = On\n";
echo "   log_errors = On\n";
echo "   display_errors = Off\n\n";

echo "3. After editing php.ini:\n";
echo "   - Save the file\n";
echo "   - Restart Apache in XAMPP Control Panel\n";
echo "   - Run this script again to verify\n\n";

echo "4. Test email sending:\n";
echo "   - Go to your POS system Settings → Email Settings\n";
echo "   - Configure SMTP settings\n";
echo "   - Click 'Test Email Configuration'\n\n";

// Additional diagnostic information
echo "=== DIAGNOSTIC INFORMATION ===\n";
echo "PHP SAPI: " . php_sapi_name() . "\n";
echo "Server Software: " . ($_SERVER['SERVER_SOFTWARE'] ?? 'Unknown') . "\n";
echo "Operating System: " . php_uname() . "\n";
echo "PHP Extensions Count: " . count(get_loaded_extensions()) . "\n";
echo "Loaded Extensions: " . implode(', ', get_loaded_extensions()) . "\n\n";

// Test basic email function (if available)
if (function_exists('mail')) {
    echo "PHP mail() function: Available\n";
    echo "Note: mail() function requires local SMTP server or sendmail\n";
} else {
    echo "PHP mail() function: Not available\n";
}

echo "\n=== END OF CHECK ===\n";
?> 